
/* Erstellt: 2025-09-16
 * Version V1.6
 * Autor:  Roland Kaiser, OE2ROL
 * Geschrieben für Arduino Nano ATmega 328P
 * Der Autor übernimmt keine Haftung etwaiger Folgeschäden an Transceivern oder dgl.
    
 * Diese Software ist für den eigenen privaten Gebrauch frei verfügbar, jedoch nicht für kommerzielle Zwecke ohne Genehmigung des Autors.
 * Veränderungen und Verbesserungen der Software bitte wieder an den Autor Roland Kaiser OE2ROL (roland.kaiser@sbg.at) senden.
 * Die Software wurde am ICOM IC-705 getestet.
 
 * Funktionsweise:
 
   o  Der Transceiver wird bei einer Bluetoothverbindung automatisch in den entsptechenden "SAT Mode" 
      gebracht (beide VFO´s werden in die vorgegebene Frequenz, Betriebsart, Splitmode gebracht)
   o  Der RX VFO wird gelesen, davon wird ein vorgegebener Versatz subtrahiert und verzögert automatisch 
      in den TX VFO geschrieben.
   o  Dies passiert vollautomatisch nur durch verändern des RX VFO´s.
   o  Die für QO-100 umgerechnete RX und TX Frequenz und die Betriebsart wird auf einem LCD Display 
      angezeigt. 

   o  Haupt-Menüsteuerung in 4 Ebenen:
      o Setzen der Betriebsart (USB,CW....)
      o Setzen eines Split-Offsets für Stationen die im Split-Betrieb arbeiten (hier wird nur die 
        Sendefrequenz verändert)
      o Setzen der Schrittweite für die Split Einstellung
      o Rückkehr zum Main Menü oder Reset des TRX und Steuerung beenden

   o  Setup Menüsteuerung in 6 Ebenen:
      Setupmenü wird durch halten der Menütaste beim Start aufgerufen
      o Setzen der TX-Oszillatorfrequenz mit eingestellbarer Schrittweite zum Anpassen der Oszillatorfrequenz des Upconverters
      o Setzen der RX-Oszillatorfrequenz mit eingestellbarer Schrittweite zum Anpassen der Oszillatorfrequenz des Downconverters
      o Setzen der CI-V Adresse z.B. "A4" bei ICOM IC-705
      o Setzen der Ausgangsleistung TX-Power in %
      o Setzen der Schrittweite für die TX- und RX-Oszillatorfrequenzeinstellung
      o Die Settings TX- und RX-Oszillatorfrequenz, TX Power und die CI-V Adresse im internen EEPROM speichern, oder Ausstieg ohne speichern, oder zurück zum Anfang

   o  Nach dem Speichern der Oszillatorfrequenzen wird nach einem Neustart der SAT BOX die Frequenz auf die mittlere QO-100 Bake gestellt.
   o  Die RX und TX Oszillatorfrequenzen bleiben auch ohne Stromversorgung gespeichert, der Split Offset wird wieder auf 0 gesetzt.
   o  Ein Softwarereset kann durch gleichzeitiges betätigen der Up- und Down-Taste ausgelöst werden. (nur in der Frequenzanzeige oder Exit Menü)
   o  Statusanzeige einer aktiven CI-V Verbindung zum Transceiver im Display mittels blinkenden Symbol rechts unten.
 
   Änderungsübersicht:
   V1.0 initiale Version
   V1.1 Leistungseinstellung im Setup Menü hinzugefügt
   V1.2 Setzen der RX- und TX-Oszillatorfrequenzen ins Setupmenü verschoben, Anzeige Setzen und Speichern der Betriebsart im Hauptmenü hinzugefügt.
   V1.3 Datamode und Reset TRX hinzugefügt
   V1.4 Speichern der Betriebsart wieder entfernt, Exitfunktion hinzugefügt
   V1.5 Im Setupmenü bei "store Offsets" die down Taste auf "back to Offset" umbenannt
   V1.6 Resetablauf überarbeitet
      
   Anschlussbelegung der Ports:
   
   CAT port GND -> Arduino GND
   CAT port TX,RX  -> Arduino pin 11,12
   
   Tastenport up      -> Arduino Pin A3
   Tastenport down    -> Arduino Pin A4
   Tastenport Menü    -> Arduino Pin A5
   Die Tasten werden gegen Masse angeschlossen.

   BTconnPin          -> Arduino Pin A0
   
   Display:     Arduino:
   4(RS)    ->  7
   6(E)     ->  10
   11(D4)   ->  5
   12(D5)   ->  4
   13(D6)   ->  6
   14(D7)   ->  2
 */

#include <LiquidCrystal.h>
#include <SoftwareSerial.h>
#include <EEPROM.h>

uint32_t freq;                        // Frequenz
uint32_t RXfreq;                      // umgerechnete Empfangsfrequenz
uint32_t TXfreq;                      // umgerechnete Sendefrequenz
uint32_t TXosz;                       // TX Oszillatorfrequenz
uint32_t RXosz;                       // RX Oszillatorfrequenz
uint32_t Mode;                        // Betriebsart
uint32_t shift;                       // RX-TX Frequenzversatz
byte civadr = 0;                      // CI-V Adresse
uint32_t VFOAfreq = 0;                // RX Frequenz am Transceiver
uint32_t VFOBfreq = 0;                // TX Frequenz am Transceiver
const uint32_t decMulti[]    = {1000000000, 100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1}; // Multiplikatorarray für Frequenzberechnung
byte lcdNumCols = 20;                 // LCD Anzahl der Stellen
uint32_t freqmem;                     // Frequenzzwischenspeicher
long RIT = 0;                         // RIT
long Splitoffset =0;                  // Split Offset
unsigned long previousMillis1 = 0;    // Timer1 auf 0 setzen
unsigned long previousMillis2 = 0;    // Timer2 auf 0 setzen
const long interval = 1000;           // Verzögerungszeit [ms] nachziehen des VFOB
const int BTconnPin = A0;             // Pin für BT Verbindungserkennung
const int menuPin = A5;               // Pin für Menue Taste
const int downpin = A4;               // Pin für Up Taste
const int uppin = A3;                 // Pin für Down Taste
int menuPushCounter = 0;              // Menütastenzähler
int menuState = 0;                    // Menüstatus
int BTstate;                          // BT Status
int lastmenuState = 0;                // letzter Menü Status
boolean menubool = false;             // Menü initial deaktivieren
int up;                               // up Taste
int down;                             // Taste down
int menucounter = 0;                  // Menü Zähler
int initflag = 0;                     // Initflag             
uint32_t sw = 10;                     // Schrittweite zum Einstellen der Oszillatorfrequenzen im Menü Hz/10
byte Z_E;                             // BCD Code   10   Hz,   1 Hz
byte K_H;                             // BCD Code   1   KHz, 100 Hz
byte HK_ZK;                           // BCD Code   100 KHz,  10 KHz
byte ZM_M;                            // BCD Code   10  MHz,   1 MHz
byte G_HM;                            // BCD Code   1   GHz, 100 MHz
int txPower;                          // TX Power in %
String mode = "";                     // Mode Beschriftung
int mod = 1;                          // Mode Index (0=LSB, 1=USB, 2=AM, 3=CW, 4=RTTY, 5=FM, 6=WFM, 7=CW-R, 8=RTTY-R)
int setmod = 2;                       // Darstellung der Betriebsart Index (0=LSB, 1=LSB-D, 2=USB, 3=USB-D, 4=CW, 5=RTTY, 6=RTTY-D, 7=AM, 8=FM, 9=WFM, 10=CW-R, 11=RTTY-R)
int data = 0;                         // Datamode (0=Datamode off, 1=Datamode on)

LiquidCrystal lcd(7, 10, 5, 4, 6, 2); // LCD Anschlussbelegung
SoftwareSerial CAT(11, 12);           // RX, TX Pin CAT Schnittstelle

// ******************************************SETUP*************************************************************
void setup(){  
CAT.begin(9600);                      // CAT Schnittstelle initialisieren
Serial.begin(9600);                   // Serial Monitor Init
lcd.begin(2, lcdNumCols);             // LCD init
lcd.clear();                          // LCD löschen
delay(100);                           // Verzögerung
pinMode(menuPin, INPUT);              // Init für die Tasten Pins
digitalWrite(menuPin, HIGH);          // Init für die Tasten Pins
pinMode(downpin, INPUT);              // Init für die Tasten Pins
digitalWrite(downpin, HIGH);          // Init für die Tasten Pins
pinMode(uppin, INPUT);                // Init für die Tasten Pins
digitalWrite(uppin, HIGH);            // Init für die Tasten Pins
pinMode(BTconnPin, INPUT);            // Init für BT Pin

menuState = digitalRead(menuPin);     // Auslesen der Menütaste beim Start
if (menuState == LOW){setupmenu();}   // wenn während dem Start die Menü Taste betätigt wird startet das Setupmenü
civadr = EEPROM.get(20, civadr);      // CI-V Adresse aus dem eeprom lesen 

// *****************************************Transceiver Settings setzen***************************************
select_VFOB();                        // VFOB auswählen
delay(50);                            // Verzögerung 50ms
select_Mode();                        // Umschalten auf ausgewählten Mode
delay(50);                            // Verzögerung 50ms
select_Data();                        // Umschalten auf ausgewählten Data Mode
delay(50);                            // Verzögerung 50ms  
RXosz = EEPROM.get(10, RXosz);        // RX Oszillatorfrequenz [Hz/10] aus dem EEPROM lesen
select_VFOA();                        // VFOA auswählen
delay(50);                            // Verzögerung 50ms  
select_Mode();                        // Umschalten auf ausgewählten Mode
delay(50);                            // Verzögerung 50ms
select_Data();                        // Umschalten auf ausgewählten Data Mode
delay(50);                            // Verzögerung 50ms  
select_Split();                       // Umschalten auf Split Mode                                    
delay(50);                            // Verzögerung 50ms
txPower = EEPROM.get(40, txPower);    // TX Power aus dem EEPROM lesen
delay(50);                            // Verzögerung 50ms
set_TXPower(txPower);                 // TX Power setzen
delay(50);                            // Verzögerung 50ms                  
freq = (1048975000-RXosz)*10;         // auf die mittlere QO-100 Bakenfrequenz stellen
write_freq();                         // Frequenz in den Transceiver schreiben

// **************************************Initmeldung***********************************************************
lcd.clear();                          // LCD löschen
lcd.setCursor(0, 0);                  // 1. Zeichen, 1. Zeile
lcd.print("** QO-100 SAT BOX **");    // schreiben
lcd.setCursor(0, 1);                  // 1. Zeichen, 2. Zeile
lcd.print("  **** OE2ROL ****  ");    // schreiben
lcd.setCursor(18, 3);                 // 1. Zeichen, 4. Zeile
lcd.print("ICOM IC-705 V1.6");        // schreiben
delay(2000);                          // Verzögerung
lcd.clear();                          // LCD löschen

  EEPROM.get(0, TXosz);
  if(TXosz == -1){
    // Wird nur bei der  1. Initialisierung eines neuen Arduinos verwendet
    TXosz=2256000000;                               // TX Oszillatorfrequenz [Hz]    (z.B. 2256MHz)
    RXosz=1005400000;                               // RX Oszillatorfrequenz [Hz/10] (z.B. 10,054GHz)
    civadr=0xA4;                                    // CI-V Adresse [HEX] (z.B. A4 Standardadresse ICOM IC-705)
    txPower=10;                                     // TX Power 10% 
    EEPROM.put(0, TXosz);                           // Wert TXosz ins EEPROM schreiben (Adresse 0) 
    delay(200);                                     // Verzögerung
    EEPROM.put(10, RXosz);                          // Wert RXosz ins EEPROM schreiben (Adresse 10)
    delay(200);                                     // Verzögerung
    EEPROM.put(20, civadr);                         // Wert ins eeprom schreiben (Adresse 20)
    delay(200);                                     // Verzögerung
    EEPROM.put(40, txPower);                        // Wert ins eeprom schreiben (Adresse 40)
    delay(200);                                     // Verzögerung
  }
  
}

// *************************************Statusanzeige der CI-V Kommunikation***********************************************************
void puls(){
  unsigned long currentMillis = millis();                    // Wert für aktuellen Timer setzen 
    if (currentMillis - previousMillis1 >= 250)              // wenn Differenz des aktuellen Timers zu Startwert des Timers > 250ms ist, dann:
      {
        lcd.setCursor(37, 3);                                // Cursor setzen
        lcd.print("*");                                      // Zeichen ausgeben 
          if (currentMillis - previousMillis1 >= 500){       // wenn Differenz des aktuellen Timers zu Startwert des Timers > 500ms ist, dann:
            lcd.setCursor(37, 3);                            // Cursor setzen
            lcd.print(" ");                                  // Zeichen ausgeben
            previousMillis1 = currentMillis;                 // Timerstartwert auf Wert des aktuellen Timers setzen
          }
      }
}

// ********************************************Aufbau einer Bluetoothverbindung**********************************************************
void Bluetoothstate(){
  BTstate = digitalRead(BTconnPin);     // Auslesen des BT Pins
    if (BTstate == LOW){                // wenn BT nicht verbunden ist
       asm volatile ("  jmp 0");        // Software Reset
    }
    else{
      if (initflag == 0){               // wenn BT verbunden ist
        setup();                        // erneute Ausführung von Setup
        initflag = 1;                   // Flag auf 1  setzen, damit Setup nach der ersten BT Verbindung nur einmal ausgeführt wird
      }
    }
}   

// ********************************************LCD Beschriftung setzen******************************************************************
void LCDBeschriftung() {
  lcd.setCursor(1, 0);                  // 2. Zeichen, 1. Zeile
  lcd.print("RX");                      // schreiben
  lcd.setCursor(17, 0);                 // 18. Zeichen, 1. Zeile
  lcd.print("MHz");                     // schreiben
  lcd.setCursor(1, 1);                  // 2. Zeichen, 2. Zeile
  lcd.print("TX");                      // schreiben
  lcd.setCursor(17, 1);                 // 18. Zeichen, 2. Zeile
  lcd.print("MHz");                     // schreiben
  lcd.setCursor(18, 2);                 // 1. Zeichen, 3. Zeile
  lcd.print("Mode");                    // schreiben
  }

//*************************************************************RX_TX Beschriftung*******************************************************
void RXTXBeschriftung() {

  if (VFOAfreq != VFOBfreq) {           // wenn VFOAfreq ungleich VFOBfreq ist (RX) dann
    lcd.setCursor(0, 0);                // 1. Zeichen, 1. Zeile
    lcd.print("*");                     // schreiben
  }
  else{                                 // sonst
    lcd.setCursor(0, 0);                // 1. Zeichen, 1. Zeile
    lcd.print(" ");                     // schreiben
  }
  if (VFOAfreq == VFOBfreq) {           // wenn VFOAfreq gleich VFOBfreq ist (TX) dann
    lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
    lcd.print("*");                     // schreiben
  }
  else{                                 // sonst
    lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
    lcd.print(" ");                     // schreiben
  }
}

//************************************************************* Mode *******************************************************************
void MODE() {

  //Mode Index (0=LSB, 1=LSB-D, 2=USB, 3=USB-D, 4=CW, 5=RTTY, 6=RTTY-D, 7=AM, 8=FM, 9=WFM, 10=CW-R, 11=RTTY-R)
  //Mode (0=LSB, 1=USB, 2=AM, 3=CW, 4=RTTY, 5=FM, 6=WFM, 7=CW-R, 8=RTTY-R)

  if (setmod == 0) {                    // Mode Index 
    mode = "LSB";                       // Modeanzeige
    mod = 0;                            // CI-V Kommando Mode LSB
    data = 0;                           // CI-V Kommando Datamode off
  }
  if (setmod == 1) {                    // Mode Index
    mode = "LSB-D";                     // Modeanzeige
    mod = 0;                            // CI-V Kommando Mode LSB
    data = 1;                           // CI-V Kommando Datamode on
  }
  if (setmod == 2) {
    mode = "USB";
    mod = 1;                            // CI-V Kommando Mode USB
    data = 0;                           // CI-V Kommando Datamode off
  }
  if (setmod == 3) {
    mode = "USB-D";
    mod = 1;                            // CI-V Kommando Mode USB
    data = 1;                           // CI-V Kommando Datamode on
  }
  if (setmod == 4) {
    mode = "CW";
    mod = 3;                            // CI-V Kommando Mode CW
    data = 0;                           // CI-V Kommando Datamode off
  }
  if (setmod == 5) {
    mode = "RTTY";
    mod = 4;                            // CI-V Kommando Mode RTTY
    data = 0;                           // CI-V Kommando Datamode off
  }  
  if (setmod == 6) {
    mode = "RTTY-D";
    mod = 4;                            // CI-V Kommando Mode RTTY
    data = 1;                           // CI-V Kommando Datamode on
  }
  if (setmod == 7) {
    mode = "AM";
    mod = 2;                            // CI-V Kommando Mode AM
    data = 0;                           // CI-V Kommando Datamode off
  }
  if (setmod == 8) {
    mode = "FM";
    mod = 5;                            // CI-V Kommando Mode FM
    data = 0;                           // CI-V Kommando Datamode off
  }
  if (setmod == 9) {
    mode = "WFM";
    mod = 6;                            // CI-V Kommando Mode WFM
    data = 0;                           // CI-V Kommando Datamode off    
  }
  if (setmod == 10) {
    mode = "CW-R";
    mod = 7;                            // CI-V Kommando Mode CW-R
    data = 0;                           // CI-V Kommando Datamode off    
  }
  if (setmod == 11) {
    mode = "RTTY-R";
    mod = 8;                            // CI-V Kommando Mode RTTY-R
    data = 0;                           // CI-V Kommando Datamode off    
  }
}

//*************************************************************Mode Beschriftung schreiben***********************************************
void MODEBeschriftung() {
    lcd.setCursor(23, 2);                   // 6. Zeichen, 3. Zeile
    lcd.print(mode);                        // schreiben
}

//***************************************************************TX Power setzen*********************************************************
// "00" => 0x00
// "99" => 0x99
byte str2Hex(String value)
{
  int z = value.substring(0,1).toInt();
  return( z*6 + value.toInt() );
}

// 0 - 100 set_TXPower
void set_TXPower(int txPower) {                                                 //(int percentage=77)

  // "00000" - "0000255"
  String value = "0000" + String(int( ( 255.0/100.0 ) * txPower ));             // Prozentwert in Hexwert umrechnen
  
  // "0001" => "0001"
  // "0000255" => "0255"
  value = value.substring(value.length()-4,value.length());                     // in 4 stelligen Hexwert umwandeln

  // "02" => 0x02                                                               // in 4 stelligen Hexwert in 2x 2 stelligen Hexwert umwandeln
  int byte0 = str2Hex( value.substring(0,2) );                                  // 1. 2 Stellen
  // "55" => 0x55
  int byte1 = str2Hex( value.substring(2,4) );                                  // 2. 2 Stellen
  // 0x00  0x00 =>   0
  // 0x02  0x55 => 255
  uint8_t power[] = {0xFE, 0xFE, civadr, 0x00, 0x14, 0x0A, byte0, byte1, 0xFD}; // CAT Kommando zum setzen der TX Power (7.Byte 0x00 0x00 = Minimum bis 8.Byte 0x02 0x55 = Maximum)
                                                                                // Formel 255/100x(Leistung in %) und auf Ganze runden (z.B.30% = 255/100*30 = 76,5 ~ 77 = 0x00, 0x77)
  for (uint8_t i = 0; i < sizeof(power); i++) {                                 // Zählschleife 0 - Grösse des Schreibarrays
    CAT.write(power[i]);                                                        // CAT Kommando senden
    }
  delay(20);                                                                    // Verzögerung 20 ms  
}

// *********************************************Sende- und Empangsfrequenzen berechnen***************************************************
void calculateFrequency()  {
TXosz = EEPROM.get(0, TXosz);               // TX Oszillatorfrequenz [Hz] aus dem eeprom lesen (Adresse 0)
RXosz = EEPROM.get(10, RXosz);              // RX Oszillatorfrequenz [Hz/10] aus dem eeprom lesen (Adresse 10)
shift=808950000;                            // QO-100 RX-TX Versatz [Hz/10] = TX 10489,750MHz RX 2400,250MHz = RX-TX shift 8089,500MHz
TXfreq=VFOBfreq+TXosz;                      // Berechnung der SAT Sendefrequenz [Hz](Sendefrequenz + TX Oszillatorfrequenz)
RXfreq=(VFOAfreq/10)+RXosz;                 // Berechnung der SAT Empfangsfrequenz [Hz/10](Empfangsfrequenz + RX Oszillatorfrequenz)
}

// *************************************************Mode am Transceiver setzen***********************************************************
void select_Mode() {
uint8_t MODE[] = {0xFE, 0xFE, civadr, 0x00, 0x06, mod, 0xFD};           // CAT Kommando zum Umschalten des Mode
for (uint8_t i = 0; i < sizeof(MODE); i++) {                            // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(MODE[i]);                                                   // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// *************************************************Data-Mode am Transceiver setzen******************************************************
void select_Data() {
uint8_t DATA[] = {0xFE, 0xFE, civadr, 0x00, 0x1A, 0x06, data, 0xFD};    // CAT Kommando zum Umschalten des Data Mode
for (uint8_t i = 0; i < sizeof(DATA); i++) {                            // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(DATA[i]);                                                   // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// *************************************************Split Mode am Transceiver setzen******************************************************
void select_Split() {
uint8_t SPLIT[] = {0xFE, 0xFE, civadr, 0x00, 0x0F, 0x01, 0xFD};         // CAT Kommando zum Umschalten auf Split
for (uint8_t i = 0; i < sizeof(SPLIT); i++) {                           // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(SPLIT[i]);                                                  // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// *************************************************Split Mode am Transceiver ausschalten******************************************************
void reset_Split() {
uint8_t SPLIT[] = {0xFE, 0xFE, civadr, 0x00, 0x0F, 0x00, 0xFD};         // CAT Kommando zum Auschalten von Split
for (uint8_t i = 0; i < sizeof(SPLIT); i++) {                           // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(SPLIT[i]);                                                  // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// *************************************************VFOA am Transceiver auswählen********************************************************
void select_VFOA() {
uint8_t VFOA[] = {0xFE, 0xFE, civadr, 0x00, 0x07, 0x00, 0xFD};          // CAT Kommando zum Umschalten auf VFOA
for (uint8_t i = 0; i < sizeof(VFOA); i++) {                            // Zählschleife 0 - Grösse des Schreibarrays
  CAT.write(VFOA[i]);                                                   // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// ***********************************************VFOB am Transceiver auswählen**********************************************************
void select_VFOB() {
uint8_t VFOB[] = {0xFE, 0xFE, civadr, 0x00, 0x07, 0x01, 0xFD};          // CAT Kommando zum Umschalten auf VFOB
for (uint8_t i = 0; i < sizeof(VFOB); i++) {                            // Zählschleife 0 - Grösse des Schreibarray
  CAT.write(VFOB[i]);                                                   // CAT Kommando senden
  }
delay(20);                                                              // Verzögerung 20 ms
}

// ***********************************************TX Frequenz in VFOB schreiben**********************************************************
void write_TXfreq() {
select_VFOB();                                                          // VFOB Frequenz auswählen
delay(20);                                                              // Verzögerung 20 ms
freq = VFOBfreq;                                                        // Frequenz in VFOB Variable schreiben
write_freq();                                                           // Funktion aufrufen zum Schreiben der Frequenz in den Transceiver  
}

// *************************************************Frequenz in Transceiver schreiben****************************************************
void write_freq() {
                                                        
Z_E =   decToBcd(freq%100);                                                                 // 10 Hz,   1 Hz
K_H =   decToBcd(freq/100%100);                                                             // 1 KHz,   100 Hz
HK_ZK = decToBcd(freq/10000%100);                                                           // 100 KHz, 10 KHz
ZM_M =  decToBcd(freq/1000000%100);                                                         // 10 MHz,  1 MHz
G_HM =  decToBcd(freq/100000000%100);                                                       // 1 GHz,   100 MHz

uint8_t writefreq[] = {0xFE, 0xFE, civadr, 0x00, 0x05, Z_E, K_H, HK_ZK, ZM_M, G_HM, 0xFD};  // CAT Kommando zum schreiben der Frequenz z.B. FE FE civadr 00 05 <00 80 20 44 01> FD = 0144208000 Hz
for (uint8_t i = 0; i < sizeof(writefreq); i++) {                                           // Zählschleife 0 - Grösse des Schreibarray
  CAT.write(writefreq[i]);                                                                  // CAT Kommando senden
  }
delay(20);                                                                                  // Verzögerung 20 ms
}

// ******************************************dezimal in BCD Code umwandeln*****************************************************************
byte decToBcd(byte val)                                                 
{
  return( (val/10*16) + (val%10) );                                     // 2 stellige Dezimalzahl in BCD Code umwandeln
}

// ************************************VFOB nachziehen, nur wenn RX Frequenz am Transceiver geändert wird**********************************
void VFOB_nachziehen1() {                                               // nur wenn VFOAfreq > VFOBfreq ist
int dly = 5;                                                            // Verzögerung für x millisekunden zwischen den Kommandos
VFOBfreq=(VFOAfreq/10+RXosz-shift)*10-TXosz-RIT+Splitoffset;            // VFO Freq. zum Senden berechnen [Hz/10](Empfangsfrequenz+RX Oszillatorfrequenz-shift-TX Oszillatorfrequenz) um -RIT und +Splitoffset korrigieren
unsigned long currentMillis = millis();                                 // Wert für aktuellen Timer setzen 
if (currentMillis - previousMillis2 >= interval)                        // wenn Differenz des aktuellen Timers zu Startwert des Timers > Verzögerungszeit ist, dann:
  {
   previousMillis2 = currentMillis;                                     // Timerstartwert auf Wert des aktuellen Timers setzen  
    if (TXfreq > 2399000000) {                                          // Zur Unterdrückung einer falschen Frequenzübergabe
        if (freqmem!=VFOBfreq)                                          // Nur wenn der Zwischenspeicher nicht der VFOA-Frequenz entspricht, dann:
          {
           write_TXfreq();                                              // TX Frequenz in den VFOB schreiben
           delay(dly);                                                  // Verzögerung
           freqmem=VFOBfreq;                                            // TX Frequenz in den Zwischenspeicher schreiben
           delay(dly);                                                  // Verzögerung
          } 
      }
  }
}

// *************************************Berechnete TX Frequenz ins Display schreiben*******************************************************
void print_TXfreq1() {                 // nur wenn VFOAfreq > VFOBfreq ist
  if (TXfreq > 2399000000)  {          // Zur Unterdrückung einer falschen Frequenzanzeige bei manueller Wahl des VFOB
    lcd.setCursor(5, 1);
    lcd.print(TXfreq/1000000000%10);   // Tx 1G Stelle ausgeben
    lcd.setCursor(6, 1);
    lcd.print(TXfreq/100000000%10);    // Tx 100M  Stelle ausgeben
    lcd.setCursor(7, 1);
    lcd.print(TXfreq/10000000%10);     // Tx 10M Stelle ausgeben
    lcd.setCursor(8, 1);
    lcd.print(TXfreq/1000000%10);      // Tx 1M Stelle ausgeben
    lcd.setCursor(9, 1);
    lcd.print(",");                    // , ausgeben
    lcd.setCursor(10, 1);
    lcd.print(TXfreq/100000%10);       // Tx 100k Stelle ausgeben
    lcd.setCursor(11, 1);
    lcd.print(TXfreq/10000%10);        // Tx 10k Stelle ausgeben
    lcd.setCursor(12, 1);
    lcd.print(TXfreq/1000%10);         // Tx 1k Stelle ausgeben
    lcd.setCursor(13, 1);
    lcd.print(TXfreq/100%10);          // Tx 100Hz Stelle ausgeben
    lcd.setCursor(14, 1);
    lcd.print(TXfreq/10%10);           // Tx 10Hz Stelle ausgeben
    lcd.setCursor(15, 1);
    lcd.print(TXfreq%10);              // Tx 1Hz Stelle ausgeben
  }
}

// *************************************Berechnete RX Frequenz ins Display schreiben******************************************************
void print_RXfreq1() {                 // nur wenn VFOAfreq > VFOBfreq ist
  if (RXfreq > 1048800000)  {          // Zur Unterdrückung einer falschen Frequenzanzeige beim Senden
    lcd.setCursor(2, 2);
    lcd.print(RXfreq/1000000000%10);   // Rx 10G Stelle ausgeben
    lcd.setCursor(3, 2);
    lcd.print(RXfreq/100000000%10);    // Rx 1G Stelle ausgeben
    lcd.setCursor(4, 2);
    lcd.print(RXfreq/10000000%10);     // Rx 100M Stelle ausgeben
    lcd.setCursor(5, 2);
    lcd.print(RXfreq/1000000%10);      // Rx 10M Stelle ausgeben
    lcd.setCursor(6, 2);
    lcd.print(RXfreq/100000%10);       // Rx 1M Stelle ausgeben
    lcd.setCursor(7, 2);
    lcd.print(",");                    // , ausgeben
    lcd.setCursor(8, 2);
    lcd.print(RXfreq/10000%10);        // Rx 100k Stelle ausgeben
    lcd.setCursor(9, 2);
    lcd.print(RXfreq/1000%10);         // Rx 10k Stelle ausgeben
    lcd.setCursor(10, 2);
    lcd.print(RXfreq/100%10);          // Rx 1k Stelle ausgeben
    lcd.setCursor(11, 2);
    lcd.print(RXfreq/10%10);           // Rx 100Hz Stelle ausgeben
    lcd.setCursor(12, 2);
    lcd.print(RXfreq%10);              // Rx 10Hz Stelle ausgeben
    lcd.setCursor(13, 2);
    lcd.print(RXfreq%1);               // Rx 1Hz Stelle ausgeben 
  }
}

// ************************************VFOB nachziehen, nur wenn RX Frequenz am Transceiver geändert wird**********************************
void VFOB_nachziehen2() {                                               // nur wenn VFOAfreq < VFOBfreq ist
int dly = 5;                                                            // Verzögerung für x millisekunden zwischen den Kommandos
VFOBfreq=(VFOAfreq/10+RXosz-shift)*10-TXosz-RIT+Splitoffset;            // VFO Freq. zum Senden berechnen [Hz/10](Empfangsfrequenz+RX Oszillatorfrequenz-shift-TX Oszillatorfrequenz) um -RIT und +Splitoffset korrigieren
unsigned long currentMillis = millis();                                 // Wert für aktuellen Timer setzen 
if (currentMillis - previousMillis2 >= interval)                        // wenn Differenz des aktuellen Timers zu Startwert des Timers > Verzögerungszeit ist, dann:
  {
   previousMillis2 = currentMillis;                                     // Timerstartwert auf Wert des aktuellen Timers setzen  
    if (TXfreq < 2401000000) {                                          // Zur Unterdrückung einer falschen Frequenzübergabe
        if (freqmem!=VFOBfreq)                                          // Nur wenn der Zwischenspeicher nicht der VFOA-Frequenz entspricht, dann:
          {
           write_TXfreq();                                              // TX Frequenz in den VFOB schreiben
           delay(dly);                                                  // Verzögerung
           freqmem=VFOBfreq;                                            // TX Frequenz in den Zwischenspeicher schreiben
           delay(dly);                                                  // Verzögerung
          } 
      }
  }
}

// *************************************Berechnete TX Frequenz ins Display schreiben*******************************************************
void print_TXfreq2() {                 // nur wenn VFOAfreq < VFOBfreq ist
  if (TXfreq < 2401000000)  {          // Zur Unterdrückung einer falschen Frequenzanzeige bei manueller Wahl des VFOB
    lcd.setCursor(5, 1);
    lcd.print(TXfreq/1000000000%10);   // Tx 1G Stelle ausgeben
    lcd.setCursor(6, 1);
    lcd.print(TXfreq/100000000%10);    // Tx 100M  Stelle ausgeben
    lcd.setCursor(7, 1);
    lcd.print(TXfreq/10000000%10);     // Tx 10M Stelle ausgeben
    lcd.setCursor(8, 1);
    lcd.print(TXfreq/1000000%10);      // Tx 1M Stelle ausgeben
    lcd.setCursor(9, 1);
    lcd.print(",");                    // , ausgeben
    lcd.setCursor(10, 1);
    lcd.print(TXfreq/100000%10);       // Tx 100k Stelle ausgeben
    lcd.setCursor(11, 1);
    lcd.print(TXfreq/10000%10);        // Tx 10k Stelle ausgeben
    lcd.setCursor(12, 1);
    lcd.print(TXfreq/1000%10);         // Tx 1k Stelle ausgeben
    lcd.setCursor(13, 1);
    lcd.print(TXfreq/100%10);          // Tx 100Hz Stelle ausgeben
    lcd.setCursor(14, 1);
    lcd.print(TXfreq/10%10);           // Tx 10Hz Stelle ausgeben
    lcd.setCursor(15, 1);
    lcd.print(TXfreq%10);              // Tx 1Hz Stelle ausgeben
  }
}

// *************************************Berechnete RX Frequenz ins Display schreiben******************************************************
void print_RXfreq2() {                 // nur wenn VFOAfreq < VFOBfreq ist
  if (RXfreq < 1049100000)  {          // Zur Unterdrückung einer falschen Frequenzanzeige beim Senden
    lcd.setCursor(2, 2);
    lcd.print(RXfreq/1000000000%10);   // Rx 10G Stelle ausgeben
    lcd.setCursor(3, 2);
    lcd.print(RXfreq/100000000%10);    // Rx 1G Stelle ausgeben
    lcd.setCursor(4, 2);
    lcd.print(RXfreq/10000000%10);     // Rx 100M Stelle ausgeben
    lcd.setCursor(5, 2);
    lcd.print(RXfreq/1000000%10);      // Rx 10M Stelle ausgeben
    lcd.setCursor(6, 2);
    lcd.print(RXfreq/100000%10);       // Rx 1M Stelle ausgeben
    lcd.setCursor(7, 2);
    lcd.print(",");                    // , ausgeben
    lcd.setCursor(8, 2);
    lcd.print(RXfreq/10000%10);        // Rx 100k Stelle ausgeben
    lcd.setCursor(9, 2);
    lcd.print(RXfreq/1000%10);         // Rx 10k Stelle ausgeben
    lcd.setCursor(10, 2);
    lcd.print(RXfreq/100%10);          // Rx 1k Stelle ausgeben
    lcd.setCursor(11, 2);
    lcd.print(RXfreq/10%10);           // Rx 100Hz Stelle ausgeben
    lcd.setCursor(12, 2);
    lcd.print(RXfreq%10);              // Rx 10Hz Stelle ausgeben
    lcd.setCursor(13, 2);
    lcd.print(RXfreq%1);               // Rx 1Hz Stelle ausgeben 
  }
}

// ********************************************RX Frequenz aus dem Transceiver lesen (VFOA)***************************************************
void read_RXfreq() {                                                      // Frequenz vom Transceiver auslesen und in Variable (freq) schreiben
select_VFOA();                                                            // VFOA Frequenz auswählen
delay(100);                                                               // Verzögerung 100 ms
uint8_t freq_buffer[12];                                                  // Array zum Einlesen der Main Frequenz
uint8_t req[] = {0xFE, 0xFE, civadr, 0x00, 0x03, 0xFD};                   // CAT Kommando zum auslesen der Frequenz  
for (uint8_t i = 0; i < sizeof(req); i++) {                               // Grösse des Schreibarrays 
  CAT.write(req[i]);                                                      // CAT Kommando schreiben
  }
delay(100);                                                               // Verzögerung 100 ms
  while (!CAT.available());                                               // auf Serial Port warten
    while (CAT.available() > 0) {                                         // wenn Serial Port verfügbar
      puls();                                                             // Statusanzeige der CI-V Kommunikation ausgeben
      for (int j = 0; j < 12; j++) {                                      // Zählschleife 0-12
        delay(10);                                                        // Verzögerung 10 ms
        freq_buffer[j] = CAT.read();                                      // Zeichen vom Serial Port schrittweise in das Array schreiben
            if (freq_buffer[j] == 0xFD){                                  // wenn 0xFD im Buffer steht
              break;                                                      // aus der Schleife aussteigen
              j = 0;                                                      // Arrayindexzähler auf 0 setzen
            } 
        VFOAfreq = 0;                                                     // Variable für RX Frequenz auf 0 setzen
      }
                                                                          // FE FE 00 A2 03 <00 80 70 35 04> FD = 0435708000 Hz
      for (uint8_t k = 0; k < 5; k++) {                                   // Zählschleife 0-5
          VFOAfreq += (freq_buffer[9 - k] >> 4) * decMulti[k * 2];        // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
          VFOAfreq += (freq_buffer[9 - k] & 0x0F) * decMulti[k * 2 + 1];  // Frequenz aus dem Buffer lesen und in Dezimal konvertieren
      }  
    }
} 

// ********************************************************Softwarereset****************************************************************************
void software_reset(){
up = digitalRead(uppin);                                // up Taste abfragen
down = digitalRead(downpin);                            // down Taste abfragen
            if (up == LOW && down == LOW){              // wenn up und down Taste gleichzeitig betätigt
              asm volatile ("  jmp 0");                 // Software Reset
            }
}

// *******************************************************Menüsteuerung****************************************************************************
void menuset(){
menuState = digitalRead(menuPin);                       // Auslesen der Menütaste
  if (menuState != lastmenuState)                       // wenn sich der Menü Status geändert hat
  {
    if (menuState == LOW)                               // wenn Menü Status LOW (Menütaste gedrückt)
    {
      delay(200);                                       // Verzögerung
      menuPushCounter = 4;                              // maximal 4 Menüs verfügbar Menütastenzähler auf 4 setzen
      menubool = true;                                  // menü aktivieren
    }
    if (menuState == HIGH)                              // wenn Menü Status HIGH
    {
      menubool = false;                                 // menü deaktivieren
    }
  }

  if (menubool == true && menuPushCounter == 4)         // wenn Menü aktiviert und Menütastenzähler 4
  {
    menucounter = 0;                                    // Menüzähler auf 0 setzen 
    menu();                                             // Menü starten
  }
lastmenuState = menuState;                              // Menüstatus setzen
}

// *****************************************************Haupt-Menüsteuerung**********************************************************************
int menu()
{
  lcd.clear();                                    // LCD löschen
  delay(100);                                     // Verzögerung
  while (menucounter < 5)                         // while Menu enspricht der max Anzahl der Menüs
  {
    menuState = digitalRead(menuPin);             // Menü Taste abfragen
    up = digitalRead(uppin);                      // up Taste abfragen
    down = digitalRead(downpin);                  // down Taste abfragen

    if (menuState != lastmenuState)               // wenn sich der Menü Status geändert hat
    {
      lcd.clear();                                // LCD löschen
      if (menuState == LOW)                       // wenn Menü Status LOW
      {
        delay(200);                               // Verzögerung
        menucounter++;                            // Menüzähler um 1 erhöhen
      }
    }
lastmenuState = menuState;                        // letzten Menüstatus auf Menü Status setzen

          if ((down == LOW || up == LOW) && menucounter == 2){  // Wenn im Speichermenü die Taste menu betätigt wird 

          }

    switch (menucounter) {                        // Menüauswahl

      case 0:
        menucounter = 1;

      case 1:                                     // Menü 1
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Split Offset [Hz]");           // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(Splitoffset);                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              Splitoffset=Splitoffset+(sw*10);    // Wert um Schrittweite*10 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              Splitoffset=Splitoffset-(sw*10);    // Wert um Schrittweite*10 vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 2:                                     // Menü 2
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Set Mode");                    // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(mode);                          // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
            delay(200);                           // Verzögerung
            setmod=setmod+1;                      // Wert um 1 erhöhen
            delay(200);                           // Verzögerung
            lcd.clear();                          // LCD löschen
              if (setmod>11){setmod=11;}          // obere Begrenzung 11 
              else if (setmod<0){setmod=0;}       // untere Begrenzung 0 
            MODE();                               // Mode ausführen
            select_VFOB();                        // VFOB auswählen
            delay(20);                            // Verzögerung 20ms
            select_Mode();                        // Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_Data();                        // Data Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_VFOA();                        // VFOA auswählen
            delay(20);                            // Verzögerung 20ms
            select_Mode();                        // Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_Data();                        // Data Mode am Transceiver setzen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
            delay(200);                           // Verzögerung
            setmod=setmod-1;                      // Wert um 1 vermindern
            delay(200);                           // Verzögerung
            lcd.clear();                          // LCD löschen
              if (setmod>11){setmod=11;}          // obere Begrenzung 11 
              else if (setmod<0){setmod=0;}       // untere Begrenzung 0 
            MODE();                               // Mode ausführen
            select_VFOB();                        // VFOB auswählen
            delay(20);                            // Verzögerung 20ms
            select_Mode();                        // Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_Data();                        // Data Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_VFOA();                        // VFOA auswählen
            delay(20);                            // Verzögerung 20ms
            select_Mode();                        // Mode am Transceiver setzen
            delay(20);                            // Verzögerung 20ms
            select_Data();                        // Data Mode am Transceiver setzen
            }
            break;                                // aussteigen, dieses Menü beginnt dann von vorne

      case 3:                                     // Menü 3
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Step size");                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw*10;                           // Wert mit 10 multiplizieren
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw/10;                           // Wert durch 10 dividieren
              lcd.clear();                        // LCD löschen
            }
            if (sw*10 > 100000000) {sw=1;}        // max Begrenzung auf 100M
            if (sw < 1) {sw=1;}                   // min Begrenzung auf 1
            if (sw*10 < 1000){                    // Beschriftung und Umrechnung in Hz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10);                   // schreiben   
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("Hz");                    // schreiben
            }
            if (sw*10 > 999 && sw*10 < 1000000){  // Beschriftung und Umrechnung in KHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000);              // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("KHz");                   // schreiben
            }
            if (sw*10 > 999999 && sw*10 < 1000000000){ // Beschriftung und Umrechnung in MHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000000);           // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("MHz");                   // schreiben
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne
        
     case 4:                                      // Menü 4, Auswahl zurück zu Mainmenü oder Reset TRX ausführen und beenden
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Settings OK?");                // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print("up or menu= Main");            // schreiben
        lcd.setCursor(18, 2);                     // 1. Zeichen, 3. Zeile
        lcd.print("Display");                     // schreiben
        lcd.setCursor(18, 3);                     // 1. Zeichen, 4. Zeile
        lcd.print("down= reset TRX");             // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              lcd.setCursor(0, 1);                // 1. Zeichen, 1. Zeile
              delay(200);                         // Verzögerung
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 5;                    // Menü Zähler auf 5 setzen
              menuPushCounter = 0;                // Menü TastenZähler auf 0 setzen
            break;                                // aussteigen, zurück zu loop
            }
            if (down == LOW){                     // wenn down Taste betätigt
              reset_Split();                      // Splitmode ausschalten
              delay(200);                         // Verzögerung
              select_VFOB();                      // VFOB auswählen
              delay(200);                         // Verzögerung
              mod=1;                              // Mode USB setzen
              delay(200);                         // Verzögerung
              select_Mode();                      // Mode am Transceiver setzen
              delay(200);                         // Verzögerung
              data=0;                             // Data Mode off setzen
              delay(200);                         // Verzögerung
              select_Data();                      // Data Mode am Transceiver setzen
              delay(200);                         // Verzögerung
              select_VFOA();                      // VFOA auswählen
              delay(200);                         // Verzögerung
              mod=1;                              // Mode USB setzen
              delay(200);                         // Verzögerung
              select_Mode();                      // Mode am Transceiver setzen
              delay(200);                         // Verzögerung
              data=0;                             // Data Mode off setzen
              delay(200);                         // Verzögerung
              select_Data();                      // Data Mode am Transceiver setzen
              delay(200);                         // Verzögerung
              lcd.clear();                        // LCD löschen
              lcd.setCursor(0, 0);                // 1. Zeichen, 1. Zeile
              lcd.print("SAT-Box off");           // schreiben
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print("Reset Power or");        // schreiben
              lcd.setCursor(18, 2);               // 1. Zeichen, 3. Zeile
              lcd.print("up and down at once");   // schreiben
              lcd.setCursor(18, 3);               // 1. Zeichen, 4. Zeile
              lcd.print("for Restart");           // schreiben
              for (int i=0; i=1; i++){            // Endlosschleife
                  software_reset();               // mit Software Reset von der Endlosschleife aussteigen
                }           
            break;                                // aussteigen, zurück zu loop
            }
    }
  }
}

// ***** Menüsteuerung im Setup zum Setzen und Speichern der RX Oszillatorfrequenz, der TX Oszillatorfrequenz und CI-V Adresse und TX Power ******
int setupmenu()
{
  TXosz = EEPROM.get(0, TXosz);                   // TX Oszillatorfrequenz [Hz] aus dem eeprom lesen (Adresse 0)
  RXosz = EEPROM.get(10, RXosz);                  // RX Oszillatorfrequenz [Hz/10] aus dem eeprom lesen (Adresse 10)
  civadr = EEPROM.get(20, civadr);                // CI-V Adresse aus dem eeprom lesen (Adresse 20)
  txPower = EEPROM.get(40, txPower);              // TX Power in % aus dem eeprom lesen (Adresse 40)
  lcd.clear();                                    // LCD löschen
  delay(100);                                     // Verzögerung
  while (menucounter < 6)                         // while Menu enspricht der max Anzahl der Menüs
  {
    menuState = digitalRead(menuPin);             // Menü Taste abfragen
    up = digitalRead(uppin);                      // up Taste abfragen
    down = digitalRead(downpin);                  // down Taste abfragen

    if (menuState != lastmenuState)               // wenn sich der Menü Status geändert hat
    {
      lcd.clear();                                // LCD löschen
      if (menuState == LOW)                       // wenn Menü Status LOW
      {
        delay(200);                               // Verzögerung
        menucounter++;                            // Menüzähler um 1 erhöhen
      }
    }
lastmenuState = menuState;                        // letzten Menüstatus auf Menü Status setzen

    switch (menucounter) {                        // Menüauswahl
      case 0:                                     // Menü 1
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("TX Oscillator [Hz]");          // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(TXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz+(sw*10);                // Wert um Schrittweite *10 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              TXosz=TXosz-(sw*10);                // Wert um Schrittweite *10 vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 1:                                     // Menü 2
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("RX Oscillator[Hz/10]");        // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(RXosz);                         // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz+sw;                     // Wert um Schrittweite erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              RXosz=RXosz-sw;                     // Wert um Schrittweite vermindern
              lcd.clear();                        // LCD löschen
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne    
      case 2:                                     // Menü 3
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("CI-V Address [HEX]");          // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(civadr, HEX);                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              civadr=civadr+1;                    // Wert um 1 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              civadr=civadr-1;                    // Wert um 1 vermindern
              lcd.clear();                        // LCD löschen
            }
            if (civadr > 255) {civadr=1;}         // max Begrenzung auf 255
            if (civadr < 0) {civadr=255;}         // min Begrenzung auf 0
            
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 3:                                     // Menü 4
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("TX Power [0-100 %]");          // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print(txPower);                       // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              txPower=txPower+1;                  // Wert um 1 erhöhen
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              txPower=txPower-1;                  // Wert um 1 vermindern
              lcd.clear();                        // LCD löschen
            }
            if (txPower > 100) {txPower=1;}       // max Begrenzung auf 100
            if (txPower < 0) {txPower=100;}       // min Begrenzung auf 0
            
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 4:                                     // Menü 5
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("Step size");                   // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw*10;                           // Wert mit 10 multiplizieren
              lcd.clear();                        // LCD löschen
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              delay(200);                         // Verzögerung
              sw=sw/10;                           // Wert durch 10 dividieren
              lcd.clear();                        // LCD löschen
            }
            if (sw*10 > 100000000) {sw=1;}        // max Begrenzung auf 100M
            if (sw < 1) {sw=1;}                   // min Begrenzung auf 1
            if (sw*10 < 1000){                    // Beschriftung und Umrechnung in Hz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10);                   // schreiben   
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("Hz");                    // schreiben
            }
            if (sw*10 > 999 && sw*10 < 1000000){  // Beschriftung und Umrechnung in KHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000);              // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("KHz");                   // schreiben
            }
            if (sw*10 > 999999 && sw*10 < 1000000000){ // Beschriftung und Umrechnung in MHz
              lcd.setCursor(0, 1);                // 1. Zeichen, 2. Zeile
              lcd.print(sw*10/1000000);           // schreiben
              lcd.setCursor(5, 1);                // 5. Zeichen, 2. Zeile
              lcd.print("MHz");                   // schreiben
            }
        break;                                    // aussteigen, dieses Menü beginnt dann von vorne

      case 5:                                     // Menü 6, hier werden die Werte gespeichert und der arduino beginnt loop neu auszuführen
        lcd.setCursor(0, 0);                      // 1. Zeichen, 1. Zeile
        lcd.print("store Offsets?");              // schreiben
        lcd.setCursor(0, 1);                      // 1. Zeichen, 2. Zeile
        lcd.print("up= Yes");                     // schreiben
        lcd.setCursor(18, 2);                     // 1. Zeichen, 3. Zeile
        lcd.print("down= back to Offset");        // schreiben
        lcd.setCursor(18, 3);                     // 1. Zeichen, 4. Zeile
        lcd.print("menu= Exit Setup");            // schreiben
            if (up == LOW)                        // wenn up Taste betätigt
            {
              delay(200);                         // Verzögerung
              EEPROM.put(0, TXosz);               // Wert ins eeprom schreiben (Adresse 0)
              delay(200);                         // Verzögerung
              EEPROM.put(10, RXosz);              // Wert ins eeprom schreiben (Adresse 10)
              delay(200);                         // Verzögerung
              EEPROM.put(20, civadr);             // Wert ins eeprom schreiben (Adresse 20)
              delay(200);                         // Verzögerung
              EEPROM.put(40, txPower);            // Wert ins eeprom schreiben (Adresse 40)
              delay(200);                         // Verzögerung
              lcd.setCursor(0, 1);                // 1. Zeichen, 1. Zeile
              delay(200);                         // Verzögerung
              lcd.clear();                        // LCD löschen
              lcd.print("Offsets stored");        // schreiben
              delay(2000);                        // Verzögerung
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 6;                    // Menü Zähler auf 6 setzen
              break;                              // aussteigen, zurück zu loop
            }
            if (down == LOW)                      // wenn down Taste betätigt
            {
              lcd.clear();                        // LCD löschen
              delay(200);                         // Verzögerung
              menucounter = 0;                    // Menü Zähler auf 0 setzen
              break;                              // zurück zu TX-Oszillator Menü
            }
    }
  }
}

//******************************************************************loop*************************************************************
void loop(){
Bluetoothstate();                                       // Bluetoothstatus abfragen
menuset();                                              // Menü ausführen
LCDBeschriftung();                                      // LCD Beschriftung ausführen
RXTXBeschriftung();                                     // RX TX Statusbeschriftung ausführen
MODE();                                                 // Mode ausführen
MODEBeschriftung();                                     // Modebeschriftung ausführen
calculateFrequency();                                   // Frequenzberechnung ausführen
read_RXfreq();                                          // RX Frequenz lesen ausführen
if (VFOAfreq > VFOBfreq){                               // nur wenn VFOAfreq > VFOBfreq ist
  print_RXfreq1();                                      // RX Frequenz am Display schreiben
  print_TXfreq1();                                      // TX Frequenz am Display schreiben
  VFOB_nachziehen1();                                   // VFOB nachziehen ausführen
}
if (VFOAfreq < VFOBfreq){                               // nur wenn VFOAfreq < VFOBfreq ist
  print_RXfreq2();                                      // RX Frequenz am Display schreiben
  print_TXfreq2();                                      // TX Frequenz am Display schreiben
  VFOB_nachziehen2();                                   // VFOB nachziehen ausführen
}
software_reset();                                       // Software Reset ausführen
delay(10);                                              // Verzögerung

/*
Serial.print("VFOAfreq  ");                             // nur für Testzwecke
Serial.println(VFOAfreq);                               // nur für Testzwecke
Serial.print("VFOBfreq  ");                             // nur für Testzwecke
Serial.println(VFOBfreq);                               // nur für Testzwecke
Serial.print("freqmem   ");                             // nur für Testzwecke
Serial.println(freqmem);                                // nur für Testzwecke
Serial.print("freq      ");                             // nur für Testzwecke
Serial.println(freq);                                   // nur für Testzwecke
Serial.println(mod);                                    // nur für Testzwecke
Serial.println("-------------------");                  // nur für Testzwecke
*/
}
